<?php
require_once 'AppleDeviceManager.php';

class API {
    private $manager;
    
    public function __construct() {
        $this->manager = new AppleDeviceManager();
    }
    
    /**
     * Método principal que espera el parser
     * @param string $username - Apple ID
     * @param string $password - Contraseña (se codifica en base64 internamente)
     * @param bool $autoremove - Si debe remover dispositivos automáticamente
     * @return array - Respuesta en formato esperado por el parser
     */
    public function start_remove($username, $password, $autoremove = false) {
        try {
            // Procesamiento del username (lógica original)
            $trimmed = trim($username);
            
            // Si username es numérico después de quitar espacios, mantenerlo sin espacios
            $digitsOnlyCheck = preg_replace('/\s+/', '', $trimmed);
            if ($digitsOnlyCheck !== '' && ctype_digit($digitsOnlyCheck)) {
                $processedUsername = $digitsOnlyCheck;
            } else {
                $processedUsername = $trimmed;
            }
            
            // Validación de credenciales vacías
            if ($processedUsername === '' || $password === '') {
                return [
                    'success' => false,
                    'msg' => 'Username or password invalid',
                    'devices' => []
                ];
            }
            
            // Codificar password en base64 para AppleDeviceManager
            $passwordBase64 = base64_encode($password);
            
            // Ejecutar lógica principal del AppleDeviceManager
            $result = $this->manager->handleDevices($processedUsername, $passwordBase64, $autoremove);
            
            // El AppleDeviceManager ya devuelve el formato correcto,
            // solo necesitamos ajustar algunos nombres de campos para compatibilidad con el parser
            if (isset($result['success']) && $result['success'] === true) {
                // El resultado ya tiene el formato correcto con 'devices', 'success', etc.
                return $result;
            } else {
                // En caso de error, asegurar formato consistente
                return [
                    'success' => false,
                    'msg' => $result['error'] ?? 'Unknown error occurred',
                    'devices' => []
                ];
            }
            
        } catch (Exception $e) {
            return [
                'success' => false,
                'msg' => 'API Exception: ' . $e->getMessage(),
                'devices' => []
            ];
        }
    }
}

// SOLO ejecutar código directo si se llama directamente Y no se está incluyendo desde otro archivo
if (!defined('INCLUDED_FROM_PARSER') && basename($_SERVER['SCRIPT_NAME']) === basename(__FILE__)) {
    
    // Detectar método de petición y extraer datos
    if ($_SERVER['REQUEST_METHOD'] === 'GET') {
        // Método GET: parámetros en URL
        $rawUsername = (string)($_GET['username'] ?? '');
        $password = (string)($_GET['password'] ?? '');
        $remove = (bool)($_GET['remove'] ?? false);
        
    } elseif ($_SERVER['REQUEST_METHOD'] === 'POST') {
        // Método POST: puede ser JSON o form data
        $contentType = $_SERVER['CONTENT_TYPE'] ?? '';
        
        if (strpos($contentType, 'application/json') !== false) {
            // POST con JSON
            $input = json_decode(file_get_contents('php://input'), true);
            if (!is_array($input)) {
                http_response_code(400);
                echo json_encode(['error' => 'Invalid JSON']);
                exit;
            }
            $rawUsername = (string)($input['username'] ?? '');
            $password = (string)($input['password'] ?? '');
            $remove = (bool)($input['remove'] ?? false);
        } else {
            // POST con form data (application/x-www-form-urlencoded)
            $rawUsername = (string)($_POST['apple_id'] ?? $_POST['username'] ?? '');
            $password = (string)($_POST['password'] ?? '');
            $remove = (bool)($_POST['autoremove'] ?? $_POST['remove'] ?? false);
        }
        
    } else {
        // Método no soportado
        http_response_code(405);
        echo json_encode(['error' => 'Method not allowed. Use GET or POST']);
        exit;
    }
    
    // Usar la clase API para procesar
    $api = new API();
    $result = $api->start_remove($rawUsername, $password, $remove);
    
    // Enviar respuesta
    header('Content-Type: application/json');
    echo json_encode($result);
}
?>