<?php
class AppleDeviceManager
{
    const MAX_RETRIES = 2;
    const TERMS_ACCEPTED_ERROR_CODE = 1001;
    const ICLOUD_TERMS_ID = 628501;
    const CLIENT_BUILD_NUMBER = '2532Project60';
    const CLIENT_MASTERING_NUMBER = '2532B11';
    
    private $config = [
        'stateId' => 'aa061122-526d-4f50-82b6-3ae1dc02377c',
        'widgetKey' => 'd39ba9916b7251055b22c7f910e2ea796ee65e98b2ddecea8f5dde8d9d1a815d',
        'userAgent' => 'Mozilla/5.0 (Macintosh; Intel Mac OS X 10_15_7) AppleWebKit/537.36 (KHTML, like Gecko) Chrome/135.0.0.0 Safari/537.36'
    ];

    private $session = [
        'cookies' => '',
        'endpoints' => [],
        'tokens' => [],
        'context' => []
    ];

    public function __construct()
    {
        $this->config['clientInfo'] = json_encode([
            'U' => $this->config['userAgent'],
            'L' => 'es-419',
            'Z' => 'GMT-06:00',
            'V' => '1.1',
            'F' => 'Fta44j1e3NlY5BNlY5BSs5uQ084akLJ8ijK7DddTf5.4tIV69LarUqUdHz16uBxLNlYAS753kg9ub0DK1cqvnEk_UeCx5TfBNlY5BNp55BNlan0Os5Apw.BQ4'
        ]);
    }

    private function resetSession()
    {
        $this->session = [
            'cookies' => '',
            'endpoints' => [],
            'tokens' => [],
            'context' => []
        ];
    }
    
    public function handleDevices($username, $password, $remove = false)
{
    for ($attempt = 1; $attempt <= self::MAX_RETRIES; $attempt++) {
        try {
            if ($attempt > 1) {
                $this->resetSession();
            }
            
            $this->authenticate($username, $password);
            $this->setupICloudSession();
            $this->initializeClientContext();

            $findMyDevices = $this->getDevices();
            $webDevices = $this->getWebDevices();
            
            // --- LOGGING: Guardar respuesta cruda de findMyDevices ---
            $this->logFindMyDevices($findMyDevices, $username);
            
            $removeResults = $remove ? $this->removeDevices($findMyDevices)['data'] : [];
            $removedIds = array_column($removeResults, 'device_id');
            $devicesFinal = [];

            foreach ($findMyDevices as $fm) {
                $deviceClass = $fm['deviceClass'] ?? null;
                $rawModel = $fm['rawDeviceModel'] ?? null;
                
                $device = [
                    // === IDENTIFICACIÓN BÁSICA ===
                    'id' => $fm['id'] ?? null,
                    'name' => trim($fm['name'] ?? ''),
                    'model' => $fm['deviceDisplayName'] ?? $fm['modelDisplayName'] ?? 'Unknown',
                    'device_class' => $fm['deviceClass'] ?? null,
                    'raw_model' => $fm['rawDeviceModel'] ?? null,
                    'is_mac' => $fm['isMac'] ?? false,

                    // === ESTADO DE REMOCIÓN ===
                    'was_removed' => in_array($fm['id'], $removedIds),
                    'removal_status' => in_array($fm['id'], $removedIds) ? 'REMOVED' : 'NOT_REMOVED',
                    'pending_remove' => $fm['pendingRemove'] ?? false,
                    'pending_remove_until_date' => ($fm['pendingRemoveUntilTS'] ?? 0) > 0 ? date('Y-m-d H:i:s', (int)($fm['pendingRemoveUntilTS'] / 1000)) : null,

                    // === BATERÍA ===
                    'battery_level_percent' => isset($fm['batteryLevel']) ? round($fm['batteryLevel'] * 100) . '%' : 'N/A',
                    'battery_status' => $fm['batteryStatus'] ?? 'Unknown',

                    // === LOCALIZACIÓN ===
                    'location_enabled' => $fm['locationEnabled'] ?? false,
                    'is_locating' => $fm['isLocating'] ?? false,
                    'location_coordinates' => null,
                    'location_timestamp_date' => null,

                    // === MODO PERDIDO ===
                    'lost_mode_enabled' => $fm['lostModeEnabled'] ?? false,
                    'lost_message' => isset($fm['lostDevice']) ? trim($fm['lostDevice']['text'] ?? '') : null,

                    // === IDENTIFICADORES ===
                    'udid' => null,
                    'serial' => null,
                    'imei' => null,
                    'ios_version' => null,
                    'image_url' => null
                ];

                // === PROCESAMIENTO ESPECÍFICO DE LOCALIZACIÓN ===
                if (isset($fm['location']) && is_array($fm['location'])) {
                    $device['location_coordinates'] = [
                        'latitude' => $fm['location']['latitude'] ?? null,
                        'longitude' => $fm['location']['longitude'] ?? null
                    ];
                    $device['location_timestamp_date'] = ($fm['location']['timeStamp'] ?? 0) > 0 ? date('Y-m-d H:i:s', (int)($fm['location']['timeStamp'] / 1000)) : null;
                }

                // === MATCHING CON WEB DEVICES PARA DATOS ADICIONALES ===
                $matched = false;
                foreach ($webDevices as $w) {
                    if (
                        isset($w['model'], $w['name']) &&
                        trim($w['model']) === trim($device['raw_model']) &&
                        trim($w['name']) === trim($device['name'])
                    ) {
                        $device['udid'] = $w['udid'] ?? null;
                        $device['serial'] = $w['serialNumber'] ?? null;
                        $device['imei'] = $w['imei'] ?? null;
                        $device['ios_version'] = $w['osVersion'] ?? null;
                        $device['image_url'] = $w['modelLargePhotoURL2x'] 
                            ?? "https://statici.icloud.com/fmipmobile/deviceImages-9.0/{$deviceClass}/{$rawModel}/online-infobox__2x.png";
                        $matched = true;
                        break;
                    }
                }

                // === IMAGEN POR DEFECTO SI NO HAY MATCH ===
                if (!$matched) {
                    $device['image_url'] = "https://statici.icloud.com/fmipmobile/deviceImages-9.0/{$deviceClass}/{$rawModel}/online-infobox__2x.png";
                }

                $devicesFinal[] = $device;
            }

            // === ORDENAR DISPOSITIVOS POR NOMBRE ===
            usort($devicesFinal, function($a, $b) {
                return strcmp($a['name'], $b['name']);
            });

            return [
                'success' => true,
                'action' => $remove ? 'remove' : 'info',
                'devices_count' => count($devicesFinal),
                'removed_count' => count($devicesFinal),
                'devices' => $devicesFinal
            ];
            
        } catch (Exception $e) {
            if ($attempt === self::MAX_RETRIES) {
                return $this->errorResponse($e->getMessage());
            }
        }
    }
    
    return $this->errorResponse("Error en el proceso de reintentos");
}
    
    // --- MÉTODO PARA LOGGING ---
    private function logFindMyDevices($findMyDevices, $username)
    {
        try {
            // Crear directorio si no existe
            $logDir = __DIR__;
            if (!is_dir($logDir)) {
                mkdir($logDir, 0777, true);
            }
    
            $logFile = $logDir . '/devices.log';
            
            // Preparar contenido del log
            $timestamp = date('Y-m-d H:i:s');
            $separator = str_repeat('=', 80);
            
            $logContent = "\n{$separator}\n";
            $logContent .= "TIMESTAMP: {$timestamp}\n";
            $logContent .= "USERNAME: {$username}\n";
            $logContent .= "DEVICES COUNT: " . count($findMyDevices) . "\n";
            $logContent .= "{$separator}\n";
            $logContent .= "RAW findMyDevices DATA:\n";
            $logContent .= json_encode($findMyDevices, JSON_PRETTY_PRINT | JSON_UNESCAPED_UNICODE) . "\n";
            $logContent .= "{$separator}\n\n";
    
            // Escribir al archivo (append mode)
            file_put_contents($logFile, $logContent, FILE_APPEND | LOCK_EX);
            
        } catch (Exception $e) {
            // Si falla el logging, no interrumpir el proceso principal
            error_log("Error logging findMyDevices: " . $e->getMessage());
        }
    }

    private function authenticate($username, $password)
    {
        $response = $this->makeRequest(
            'https://idmsa.apple.com/appleauth/auth/signin?widgetKey=' . $this->config['widgetKey'],
            $this->buildAuthHeaders(),
            json_encode([
                'accountName' => $username,
                'password' => base64_decode($password),
                'rememberMe' => true,
                'trustTokens' => [],
                'pause2FA' => true
            ])
        );

        if (!preg_match('/X-Apple-Session-Token:\s*(.+)/i', $response['headers'], $matches)) {
            //throw new Exception('Error de autenticación. Headers: ' . $response['headers']);
            throw new Exception('Error de autenticación');
        }

        $this->session['tokens']['webAuth'] = trim($matches[1]);
    }

    private function setupICloudSession()
    {
        $response = $this->makeRequest(
            'https://setup.icloud.com/setup/ws/1/accountLogin',
            $this->buildICloudHeaders(),
            json_encode([
                "dsWebAuthToken" => $this->session['tokens']['webAuth'],
                "extended_login" => false
            ])
        );

        $data = json_decode($response['body'], true);
        if (!is_array($data) || !isset($data['webservices']['findme']['url'], $data['webservices']['account']['url'])) {
            throw new Exception("Respuesta inválida de Apple al iniciar sesión iCloud:\n" . $response['body']);
        }

        $this->session['endpoints'] = [
            'findMy' => $data["webservices"]["findme"]["url"],
            'account' => $data["webservices"]["account"]["url"]
        ];

        $this->extractCookies($response['headers']);
    }

    private function initializeClientContext()
    {
        $response = $this->makeRequest(
            $this->session['endpoints']['findMy'] . '/fmipservice/client/web/initClient',
            $this->buildClientHeaders(),
            json_encode([
                'clientContext' => [
                    'appName' => 'iCloud Find (Web)',
                    'appVersion' => '2.0',
                    'apiVersion' => '3.0',
                    'deviceListVersion' => 1,
                    'fmly' => true,
                    'timezone' => 'US/Pacific',
                    'inactiveTime' => 77
                ]
            ])
        );

        $data = json_decode($response['body'], true);
        if (!is_array($data) || !isset($data['serverContext'])) {
            if ($this->acceptICloudTerms()) {
                throw new Exception("TERMS_ACCEPTED", self::TERMS_ACCEPTED_ERROR_CODE);
            } else {
                throw new Exception("Error Client - No se pudieron aceptar términos de iCloud");
            }
        }

        $this->session['context'] = $data['serverContext'];
    }

    private function acceptICloudTerms()
    {
        $clientId = $this->extractClientIdFromCookies();
        $prsId = $this->extractPrsIdFromCookies();
        
        if (empty($clientId) || empty($prsId)) {
            return false;
        }
        
        $url = "https://setup.icloud.com/setup/ws/1/repairDone?" . http_build_query([
            'clientBuildNumber' => self::CLIENT_BUILD_NUMBER,
            'clientMasteringNumber' => self::CLIENT_MASTERING_NUMBER,
            'clientId' => $clientId,
            'dsid' => $prsId
        ]);

        $payload = ['acceptedICloudTerms' => self::ICLOUD_TERMS_ID];

        $response = $this->makeRequest(
            $url,
            $this->buildRepairDoneHeaders(),
            json_encode($payload)
        );

        $data = json_decode($response['body'], true);
        return is_array($data) && isset($data['success']) && $data['success'] === true;
    }

    private function buildRepairDoneHeaders()
    {
        return [
            'Host: setup.icloud.com',
            'Connection: keep-alive',
            'Content-Length: 30',
            'sec-ch-ua-platform: "Windows"',
            'User-Agent: Mozilla/5.0 (Windows NT 10.0; Win64; x64) AppleWebKit/537.36 (KHTML, like Gecko) Chrome/140.0.0.0 Safari/537.36',
            'sec-ch-ua: "Chromium";v="140", "Not=A?Brand";v="24", "Google Chrome";v="140"',
            'Content-Type: text/plain;charset=UTF-8',
            'sec-ch-ua-mobile: ?0',
            'Accept: */*',
            'Origin: https://www.icloud.com',
            'Sec-Fetch-Site: same-site',
            'Sec-Fetch-Mode: cors',
            'Sec-Fetch-Dest: empty',
            'Referer: https://www.icloud.com/',
            'Accept-Encoding: gzip, deflate, br, zstd',
            'Accept-Language: es',
            'Cookie: ' . $this->session['cookies']
        ];
    }

    private function getWebDevices()
    {
        $url = "https://p38-setup.icloud.com/setup/web/device/getDevices";

        $context = $this->session['context'];
        $clientId = $context['clientId'] ?? '';
        $info = $context['info'] ?? [];
        $prsId = $context['prsId'] ?? '';

        $headers = [
            "Host: p38-setup.icloud.com",
            "Connection: keep-alive",
            'sec-ch-ua: "Not/A)Brand";v="8", "Chromium";v="126", "Google Chrome";v="126"',
            'sec-ch-ua-platform: "Windows"',
            'sec-ch-ua-mobile: ?0',
            'User-Agent: ' . $this->config['userAgent'],
            'Content-Type: text/plain',
            'Accept: */*',
            'Origin: https://www.icloud.com',
            'Sec-Fetch-Site: same-site',
            'Sec-Fetch-Mode: cors',
            'Sec-Fetch-Dest: empty',
            'Referer: https://www.icloud.com/',
            'Accept-Encoding: gzip, deflate, br',
            'Accept-Language: es-419,es;q=0.9',
            'Cookie: ' . $this->session['cookies']
        ];

        $postData = json_encode([
            "serverContext" => array_merge($context, [
                "info" => $info,
                "clientId" => $clientId,
                "prsId" => $prsId,
                "id" => "server_ctx"
            ]),
            "clientContext" => [
                "appName" => "iCloud Find (Web)",
                "appVersion" => "2.0",
                "apiVersion" => "3.0",
                "deviceListVersion" => 1,
                "fmly" => true,
                "timezone" => "US/Pacific",
                "inactiveTime" => 36739
            ]
        ]);

        $response = $this->makeRequest($url, $headers, $postData);
        $data = json_decode($response['body'], true);

        if (!is_array($data) || !isset($data['devices'])) {
            //throw new Exception("No se pudo obtener la lista de dispositivos extendida: " . substr($response['body'], 0, 200));
            throw new Exception("No se pudo obtener la lista de dispositivos extendida");
        }

        return $data['devices'];
    }

    private function getDevices()
    {
        $response = $this->makeRequest(
            $this->session['endpoints']['findMy'] . '/fmipservice/client/web/refreshClient',
            $this->buildClientHeaders(),
            json_encode([
                "serverContext" => $this->session['context'],
                "clientContext" => [
                    "appName" => "iCloud Find (Web)",
                    "appVersion" => "2.0",
                    "apiVersion" => "3.0",
                    "deviceListVersion" => 1,
                    "fmly" => true,
                    "timezone" => "US/Pacific",
                    "inactiveTime" => 953
                ]
            ])
        );

        $data = json_decode($response['body'], true);

        if (!is_array($data) || !isset($data['content'])) {
            //throw new Exception("No se pudo obtener la lista de dispositivos: " . substr($body, 0, 200));
            throw new Exception("No se pudo obtener la lista de dispositivos");
        }

        return $data['content'];
    }

    private function removeDevices($devices)
    {
        $results = [];
        foreach ($devices as $device) {
            $token = $this->getRemovalToken();
            $clientId = $this->session['context']['clientId'] ?? '';
            $prsId = $this->session['context']['prsId'] ?? '';

            $results[] = [
                'device_id' => $device['id'],
                'status' => $this->executeRemoval($device['id'], $token, $prsId, $clientId)
            ];
        }
        return $this->successResponse($results);
    }

    private function getRemovalToken()
    {
        $response = $this->makeRequest(
            $this->session['endpoints']['account'] . '/setup/ws/1/fmipWebAuthenticate',
            $this->buildRemovalHeaders('account'),
            json_encode([
                "serverContext" => $this->session['context'],
                "clientContext" => [
                    "appName" => "iCloud Find (Web)",
                    "appVersion" => "2.0",
                    "apiVersion" => "3.0",
                    "deviceListVersion" => 1,
                    "fmly" => true,
                    "timezone" => "US/Pacific",
                    "inactiveTime" => 7160
                ],
                "dsWebAuthToken" => $this->session['tokens']['webAuth'],
                "generateFmipWebShortLivedToken" => true
            ])
        );

        $data = json_decode($response['body'], true);
        return $data['tokens']['mmeFmipWebShortLivedToken'] ?? null;
    }

    private function executeRemoval($deviceId, $authToken, $prsId, $clientID)
    {
        $url = "{$this->session['endpoints']['findMy']}/fmipservice/client/web/remove?" . http_build_query([
            'clientBuildNumber'     => '2513Project44',
            'clientMasteringNumber' => '2513B20',
            'clientId'              => $clientID,
            'dsid'                  => $prsId
        ]);

        $postData = json_encode([
            'serverContext' => $this->session['context'],
            'clientContext' => [
                'appName'            => 'iCloud Find (Web)',
                'appVersion'         => '2.0',
                'apiVersion'         => '3.0',
                'deviceListVersion'  => 1,
                'fmly'               => true,
                'timezone'           => 'US/Pacific',
                'inactiveTime'       => 8011
            ],
            'device'      => $deviceId,
            'authToken'   => $authToken,
            'webTokenV2'  => true
        ]);

        $response = $this->makeRequest(
            $url,
            $this->buildRemovalHeaders('findMy'),
            $postData
        );

        foreach (array_reverse(explode("\r\n", trim($response['headers']))) as $line) {
            if (stripos($line, 'HTTP/') === 0) {
                return (strpos($line, '200') !== false) ? 'SUCCESS' : 'FAILED';
            }
        }

        return 'FAILED';
    }

    private function makeRequest($url, $headers, $payload)
    {
        $ch = curl_init();
        curl_setopt_array($ch, [
            CURLOPT_URL => $url,
            CURLOPT_RETURNTRANSFER => true,
            CURLOPT_HEADER => true,
            CURLOPT_HTTPHEADER => $headers,
            CURLOPT_POSTFIELDS => $payload,
            CURLOPT_ENCODING => '',
            CURLOPT_SSL_VERIFYPEER => false
        ]);

        $response = curl_exec($ch);
        $headerSize = curl_getinfo($ch, CURLINFO_HEADER_SIZE);

        $this->logRequest($url, $headers, $payload, substr($response, 0, $headerSize), substr($response, $headerSize));

        return [
            'headers' => substr($response, 0, $headerSize),
            'body' => substr($response, $headerSize)
        ];
    }

    private function extractCookies($headers)
    {
        preg_match_all('/Set-Cookie:\s*([^;]*)/mi', $headers, $matches);
        $this->session['cookies'] = implode('; ', $matches[1]);
    }

    private function buildAuthHeaders()
    {
        return [
            'X-Apple-Domain-Id: 3',
            'X-Apple-OAuth-State: ' . $this->config['stateId'],
            'X-Apple-I-FD-Client-Info: ' . $this->config['clientInfo'],
            'X-Apple-Widget-Key: ' . $this->config['widgetKey'],
            'Content-Type: application/json',
            'User-Agent: ' . $this->config['userAgent']
        ];
    }

    private function buildICloudHeaders()
    {
        return [
            'Host: setup.icloud.com',
            'Connection: keep-alive',
            'sec-ch-ua: "Not_A Brand";v="8", "Chromium";v="120", "Microsoft Edge";v="120"',
            'sec-ch-ua-platform: "Windows"',
            'sec-ch-ua-mobile: ?0',
            'User-Agent: ' . $this->config['userAgent'],
            'Content-Type: text/plain;charset=UTF-8',
            'Accept: */*',
            'Origin: https://www.icloud.com',
            'Sec-Fetch-Site: same-site',
            'Sec-Fetch-Mode: cors',
            'Sec-Fetch-Dest: empty',
            'Referer: https://www.icloud.com/',
            'Accept-Encoding: gzip, deflate, br',
            'Accept-Language: es,es-ES;q=0.9,en;q=0.8,en-GB;q=0.7,en-US;q=0.6',
            'Cookie: ' . $this->session['cookies']
        ];
    }

    private function buildClientHeaders()
    {
        return [
            'Connection: keep-alive',
            'sec-ch-ua: "Not/A)Brand";v="8", "Chromium";v="126", "Google Chrome";v="126"',
            'sec-ch-ua-platform: "macOS"',
            'sec-ch-ua-mobile: ?0',
            'User-Agent: ' . $this->config['userAgent'],
            'Accept: */*',
            'Origin: https://www.icloud.com',
            'Sec-Fetch-Site: same-site',
            'Sec-Fetch-Mode: cors',
            'Sec-Fetch-Dest: empty',
            'Referer: https://www.icloud.com/',
            'Accept-Language: es-419,es;q=0.9',
            'Content-Type: text/plain',
            'Accept-Encoding: gzip, deflate, br',
            'Cookie: ' . $this->session['cookies']
        ];
    }

    private function buildRemovalHeaders(string $type = 'account')
    {
        if (!in_array($type, ['account', 'findMy'])) {
            throw new InvalidArgumentException("Invalid header type: $type");
        }

        $endpointKey = $type === 'findMy' ? 'findMy' : 'account';
        $host = parse_url($this->session['endpoints'][$endpointKey], PHP_URL_HOST);

        return [
            'Host: ' . $host,
            'Connection: keep-alive',
            'User-Agent: ' . ($this->config['userAgent'] ?? 'Mozilla/5.0 (Macintosh; Intel Mac OS X 10_15_7) AppleWebKit/537.36 (KHTML, like Gecko) Chrome/135.0.0.0 Safari/537.36'),
            'Content-Type: text/plain',
            'Accept: */*',
            'Origin: https://www.icloud.com',
            'Referer: https://www.icloud.com/',
            'sec-ch-ua: "Google Chrome";v="135", "Not-A.Brand";v="8", "Chromium";v="135"',
            'sec-ch-ua-platform: "macOS"',
            'sec-ch-ua-mobile: ?0',
            'Sec-Fetch-Site: same-site',
            'Sec-Fetch-Mode: cors',
            'Sec-Fetch-Dest: empty',
            'Accept-Encoding: gzip, deflate, br, zstd',
            'Accept-Language: es-419,es;q=0.9',
            'Cookie: ' . ($this->session['cookies'] ?? $this->cookies ?? '')
        ];
    }

    private function logRequest($url, $headers, $body, $resHeaders, $resBody)
    {
        $logDir = __DIR__ . '/log';
        if (!is_dir($logDir)) {
            mkdir($logDir, 0777, true);
        }

        $logFile = $logDir . '/' . date('Y-m-d_H-i-s') . '.log';
        $log = "URL: $url\nHEADERS: " . json_encode($headers) . 
               "\nBODY: $body\nRESPONSE: $resBody\n\n";

        file_put_contents($logFile, $log);
    }

    private function successResponse($data)
    {
        return ['success' => true, 'data' => $data];
    }

    private function errorResponse($message)
    {
        return ['success' => false, 'error' => $message];
    }

    private function extractClientIdFromCookies()
    {
        if (preg_match('/X-APPLE-UNIQUE-CLIENT-ID="([^"]+)"/', $this->session['cookies'], $matches)) {
            return $matches[1];
        }
        return '';
    }

    private function extractPrsIdFromCookies()
    {
        if (preg_match('/X-APPLE-WEBAUTH-USER="v=1:s=0:d=(\d+)"/', $this->session['cookies'], $matches)) {
            return $matches[1];
        }
        return '';
    }
}
